C      *****************************************************************
C      * Subroutine Solve                                              *
C      * The goal of this routine is to find a set of values for       *
C      * component concentrations that gives the known total           *
C      * concentration of all components, assuming equilibrium.  First *
C      * it calculates dependent species concentrations from the       *
C      * current guesses for the concentrations for the components.    *
C      * Then it calculates the error between the resulting total      *
C      * component concentrations and the known total component        *
C      * concentrations.  A set of simultaneous linear equations       *
C      * describing the dependence of total component concentrations   *
C      * upon component concentration guesses is solved to find a      *
C      * corrected set of guesses for the component concentrations.    *
C      * This procedure is repeated until the calculated total         *
C      * component concentrations are close enough to the known total  *
C      * concentrations or the maximum number of iterations (set in    *
C      * this routine) is exceeded.                                    *
C      *                                                               *
C      * Variables:                                                    *
C      * Ask       -Local - Indicates whether to prompt user to quit or*
C      *                    not during each iteration. (1=prompt, 0=do *
C      *                    not prompt.)  Unitless.                    *
C      * C         -Local - Component index.  Unitless.                *
C      * Converge  -Local - Loop control variable which indicates      *
C      *                    whether solution has converged or not.     *
C      *                    (1=converged, 0=not converged.)  Unitless. *
C      * ErrTop    -Local - The convergence criterion for MaxError.    *
C      *                    If MaxError is less than ErrTop then       *
C      *                    solution has converged.  Unitless.         *
C      * GoAhead   -Output- Indicates whether any errors were found.   *
C      *                    (GoAhead=1, no errors found; GoAhead=0,    *
C      *                    errors found.)  Unitless.                  *
C      * HoldResid -Local - A vector, of size NComp, of the residual   *
C      *                    for each component. This residual is the   *
C      *                    difference between the known total         *
C      *                    concentration and the total concentration  *
C      *                    calculated from a set of component         *
C      *                    concentrations.  Units are mol.            *
C      *                    (Common block VModel, file VModel.f)       *
C      * Iter      -Local - Keeps track of the number of iterations    *
C      *                    performed.  Unitless.                      *
C      * LNum      -Input - The current layer number.  Unitless.       *
C      * MaxError  -Local - Largest ratio of residual to known         *
C      *                    concentration.  Units are mol/mol.         *
C      * MaxComp   -Local - Component corresponding to MaxError.       *
C      *                    Unitless.                                  *
C      * MaxIter   -Local - The maximum number of iterations routine   *
C      *                    can run.  Unitless.                        * 
C      * NCompSize -Input - The max number of components, used to size *
C      *                    arrays.  Unitless.                         *
C      *                    (file ArraySizes.Inc)                      *
C      * NumComp   -Input - The number of components in the system.    *
C      *                    Unitless.                                  *
C      * OutInfo   -Input - Indicates whether or not detailed          *
C      *                    information is output.  Unitless.          *
C      *                    (Common block VModel, file VModel.f)       *              
C      * Quit      -Local - Indicates whether user has chosen to quit. *
C      *                    (1=quit, 0=keep going unless converged.)   *
C      *                    Unitless.                                  *
C      * UnitNum   -Input - The output file number.  Unitless.         *
C      *****************************************************************
       SUBROUTINE SOLVE(LNUM, UNITNUM, OUTINFO, GOAHEAD, NUMCOMP)
							IMPLICIT NONE
       INCLUDE 'ARRAYSIZES.INC'
       LOGICAL OUTINFO
							INTEGER  ASK, C, CONVERGE/0/, GOAHEAD, ITER/0/, LNUM 
       INTEGER  MAXCOMP, MAXITER/500/, NUMCOMP
							INTEGER  QUIT/0/, UNITNUM
							REAL*8 ERRTOP/0.000001/, MAXERROR, HOLDRESID(NCOMPSIZE)
							
C      *---------------------------------------------------------------*
C      * Initialize variable indicating whether to prompt user to quit *
C      * or not (1 = prompt, 0 = do not prompt)                        *
C      *---------------------------------------------------------------*
       ASK = 0
	   
C      *----------------------------------------------------------------*
C      * Calculate the concentration of components controlled by phases *
C      * then given component concentrations, calculate species         *
C      * concentrations.                                                *
C      *----------------------------------------------------------------*
       CALL CALCSPECIESCONC(LNUM, GOAHEAD)
       IF (GOAHEAD.NE.1) GOTO 9999
   
C      *------------------------------------------------------------*
C      * Calc the residual between the known total concentrations   *
C      * and the concentrations based on mass balance for each comp.*
C      * Return the max error and the comp number of that error.    *
C      *------------------------------------------------------------*
       CALL CALCRESIDUAL(MAXERROR, MAXCOMP, LNUM, HOLDRESID)

C      *-----------------------------------------------------------------*
C      * Improve on component concentrations until convergence criterion *
C      * is met.                                                         *
C      *-----------------------------------------------------------------*
  100  CONTINUE   
	   
C         *----------------------------------------*
C         * Keep track of the number of iterations *
C         *----------------------------------------*
          ITER = ITER + 1
	   
C         *------------------------------------------------------*
C         * Create the Jacobian matrix of the derivatives of the *
C         * residuals.                                           *
C         *------------------------------------------------------*
          CALL JAKE(LNUM)

C         *------------------------------------------------------------*
C         * Solve the set of simultaneous equations set up in Jacobian *
C         *------------------------------------------------------------*
          CALL GAUSS(ITER,UNITNUM)
	   
C         *-----------------------------------------------------------*
C         * Update the component concentrations based on the solution *
C         * to the simultaneous equations.                            *
C         *-----------------------------------------------------------*
          CALL UPDATE(LNUM)

C         *----------------------------------------------------------*
C         * Calc the species concentrations from the component conc. *
C         *----------------------------------------------------------*
          CALL CALCSPECIESCONC(LNUM, GOAHEAD)
          IF (GOAHEAD.NE.1) GOTO 9999
  
C         *------------------------------------------------------------*
C         * Calc the residual between the analytically known concs     *
C         * and the concentrations based on mass balance for each comp.*
C         * Return the max error and the comp number of that error.    *
C         *------------------------------------------------------------*
           CALL CALCRESIDUAL(MAXERROR, MAXCOMP, LNUM, HOLDRESID)

C         *--------------------------*
C         * Output debug information *
C         *--------------------------*
C          CALL SLVOUT(ITER, MAXERROR, MAXCOMP, ASK, QUIT)
	   
C         *-----------------------*
C         * Check for convergence *
C         *-----------------------*
          IF ((ABS(MAXERROR).LT.ERRTOP).OR.(QUIT.EQ.1)) 
     >    THEN
		          CONVERGE = 1
            IF (OUTINFO) THEN
			          WRITE(UNITNUM,*) '  '
	         		 WRITE(UNITNUM,*) '*********************************'
	            WRITE(UNITNUM,*) '*** CONVERGENCE CRITERION MET ***'
			          WRITE(UNITNUM,1000) ITER
 1000        FORMAT(' ','***    ',I3,' ITERATIONS         ***')
			          WRITE(UNITNUM,1001) LNUM
 1001        FORMAT(' ','*** LAYER ',I3,'                 ***')
			          WRITE(UNITNUM,1002) MAXERROR
 1002        FORMAT(' ','*** MAXERROR',E15.4,'   ***')
			          WRITE(UNITNUM,1003) MAXCOMP
 1003        FORMAT(' ','*** MAXCOMP ',I3,'               ***')
             WRITE(UNITNUM,*) '*** RESIDUALS:                ***'
             WRITE(UNITNUM,1004) (('*** ',HOLDRESID(C)), C=1,NUMCOMP)
 1004        FORMAT(' ',A4,E15.4)
         			 WRITE(UNITNUM,*) '*********************************'
         			 WRITE(UNITNUM,*) '  '
            ENDIF
	      	  ENDIF
	      IF ((CONVERGE.EQ.0).AND.(ITER.LE.MAXITER)) GOTO 100

C      *----------------------------------------*
C      * If the model did not converge, say so. *
C      *----------------------------------------*
       IF (ITER.GE.MAXITER) THEN
			       WRITE(6,*) '  '
	       	 WRITE(6,*) '================================='
	         WRITE(6,*) '== SOIL MODEL DID NOT CONVERGE =='
			       WRITE(6,1010) ITER
 1010     FORMAT(' ','===    ',I3,' ITERATIONS         ===')
				      WRITE(6,1011) LNUM
 1011     FORMAT(' ','===    LAYER ',I3,'              ===')
          WRITE(6,*) '================================='
          WRITE(6,*) '  '

			       WRITE(UNITNUM,*) '  '
	         WRITE(UNITNUM,*) '================================='
	         WRITE(UNITNUM,*) '== SOIL MODEL DID NOT CONVERGE =='
			       WRITE(UNITNUM,1020) ITER
 1020     FORMAT(' ','===    ',I3,' ITERATIONS         ===')
			       WRITE(UNITNUM,1021) LNUM
 1021     FORMAT(' ','===    LAYER ',I3,'              ===')
	         WRITE(UNITNUM,*) '================================='
          WRITE(UNITNUM,*) '  '
	      ENDIF

C      *---------------*
C      * Escape hatch. *
C      *---------------*
 9999  CONTINUE

	      RETURN
	      END
C      *****************************************************************
C      *                    END SUBROUTINE                             *
C      *****************************************************************
